#include <mmsystem.h>
#include <stdlib.h>
#include <math.h>

// Global declarations
const int SAMPLE_FREQUENCY		= 44100;
const int SAMPLE_FREQUENCY_01	= (int)(0.01*SAMPLE_FREQUENCY);
const int BPM					= 135;
const int TEMPLATE_MELODY_LEN	= 34;
const int ROW_COUNT				= 16 * TEMPLATE_MELODY_LEN;
const int PLAY_TIME				= ROW_COUNT * 60 / ( BPM * 4 );
const int ROW_SIZE_SAMPLES		= SAMPLE_FREQUENCY * 60 / ( BPM * 4 );
const float MASTER_VOLUME		= 0.58f;
const float ROW_LEN_SEC			= 60.0f / ( BPM * 4 );
const float PI					= 3.1415926535f;
const float PI_2				= 2*PI;
HWAVEOUT	g_wo;
int			g_patternNo;
int			g_patternRowNo;
struct Instrument { 
	char note;
	char noteEnd;
	unsigned char noteLen;
	char volume;
	char oscType;
	char noteOffset;
	char attackTime;
	char decayTime;
	char sustainTime;
	char filterCutoff;
	char filterCutoffModulation;
	char filterRes;
	char echoDelay;
	char echoRepeats;
	char echoDamping;
	char wrapLevel;
	char hardLevel;
	char compressorLow;
	char compressorHigh;
};

const Instrument g_instruments[] = { { -1,2,11,86,0,0,1,22,64,70,-51,78,0,0,0,0,0,87,120 }
, { -2,0,5,6,3,0,2,19,45,77,0,92,0,0,0,0,0,0,0 }
, { -3,0,13,23,1,0,10,34,89,13,0,75,36,3,30,43,0,0,0 }
, { -3,0,27,10,0,-12,12,41,87,0,0,0,0,0,0,0,0,0,0 }
, { -4,0,10,2,1,21,11,37,84,25,44,83,10,2,26,0,0,11,92 }
, { -5,0,71,11,2,0,48,64,87,21,-15,48,30,6,89,59,0,0,0 }
, { -6,0,19,13,2,0,37,64,94,-7,0,4,22,3,58,0,0,0,0 }
, { -8,1,10,25,1,0,1,21,47,19,-12,87,0,0,0,0,0,0,0 }
, { -9,1,8,56,0,0,3,26,63,0,0,0,0,0,0,0,0,51,96 }
, { -9,1,11,64,0,0,6,17,68,0,0,0,0,0,0,0,0,0,0 }
, { -10,3,8,23,0,0,3,26,74,0,0,0,6,1,89,0,0,50,96 }
, { -11,0,27,29,2,-16,3,21,95,18,-12,87,24,4,75,0,0,0,0 }
 };
const int INSTRUMENT_COUNT = sizeof(g_instruments) / sizeof(Instrument);

const char g_bytePatterns[][16] = { {30,0,0,0,30,0,0,0,30,0,0,0,30,0,0,0},
{1,0,1,0,1,0,1,0,1,0,1,0,1,1,1,0},
{20,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
{22,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
{24,0,0,0,0,0,0,0,23,0,0,0,0,0,0,0},
{0,0,0,0,50,0,0,0,0,0,0,0,50,0,0,0},
{30,0,30,0,30,0,30,0,0,0,30,0,30,30,30,0},
{10,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0},
{12,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0},
{14,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0},
{30,0,21,0,30,0,22,0,30,0,21,0,30,0,22,0},
{30,30,0,30,31,0,30,30,0,30,31,0,32,31,30,0},
{10,12,14,16,18,20,22,24,26,28,30,32,34,36,38,40},
{42,44,46,48,50,52,54,56,58,60,62,62,62,62,62,62},
{40,40,40,40,45,40,40,40,45,40,45,40,40,40,50,45},
{40,0,41,0,40,0,41,0,40,0,41,0,40,0,42,0},
{40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40},
{8,8,7,6,20,6,7,0,0,0,0,0,0,0,0,0}};

const char g_byteSequences[][TEMPLATE_MELODY_LEN] = { {0,0,0,0,0,0,0,0,1,1,1,1,7,7,7,7,11,11,11,11,11,11,11,11,11,11,11,11,0,0,0,0,18,0},
{0,0,0,0,2,2,2,2,2,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
{0,0,0,0,8,8,9,10,8,8,9,10,0,0,0,0,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
{0,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
{3,3,4,5,3,3,4,5,3,3,4,5,3,3,4,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,12,12,12,12,12,12,12,12,12,0,0},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13,13,13,14,13,13,13,14,13,13,13,14,0,0},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,0,0},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,17,17,17,0,0},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,0},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,0}};


// Some auxiliary functions
float Sign ( float f ) {
	if ( f > 0.0f )
		return 1.0f;
	return -1.0f;
}
float CutLevel ( float x, float lvl ) {
	if ( x > lvl )
		return lvl;
	if ( x < -lvl )
		return -lvl;
	return x;
}
float buf0LPRes = 0, buf1LPRes = 0;
float LowPassResonantFilter ( float x, float f, float q ) {
	float fb = q + q / ( 1.0f - f );
	buf0LPRes = buf0LPRes + f * ( x - buf0LPRes + fb * ( buf0LPRes - buf1LPRes ) );
	buf1LPRes = buf1LPRes + f * ( buf0LPRes - buf1LPRes );
	return buf1LPRes;
}
float Sawtooth ( float x ) {
	return ( ( x - floor ( x / PI_2 ) * PI_2 ) / PI_2 - 0.5f ) * 2;
}
float Square ( float x ) {
	return Sign ( Sawtooth ( x ) );
}
float NoteToFrequency ( char note  ) {
	return PI_2 * 440.0f * powf ( 2.0f, (note-45)/12.0f );
}
char GetFixedOrStreamValue ( char val ) {
	if ( val >= 0 )
		return val;
	val = -val;
	int k = g_byteSequences[val-1][g_patternNo];
	if ( k )
		return g_bytePatterns[k-1][g_patternRowNo];
	return 0;
}

// Creates large buffer and fills it with samples. Starts playing generated sound.
// Pefroms no error checking and does not dispose resources.
void CreateAndPlaySynth ( ) {
	WAVEFORMATEX wfx = { WAVE_FORMAT_PCM, 1, SAMPLE_FREQUENCY, SAMPLE_FREQUENCY*2, 2, 16, 0 };
	waveOutOpen ( &g_wo, WAVE_MAPPER, &wfx, NULL, 0, CALLBACK_NULL );
	short* finalBuf = new short[(PLAY_TIME+20)*SAMPLE_FREQUENCY]; // We need +20 since some sounds (e.g. echoed) can last for a long time. Hope it's large enough :)
	float* data = new float[(PLAY_TIME+20)*SAMPLE_FREQUENCY];
	ZeroMemory ( data, (PLAY_TIME+20)*SAMPLE_FREQUENCY*sizeof(float) );

	// Loop through all instruments and create sound for each one of them independently
	for ( int i = 0; i < INSTRUMENT_COUNT; i++ ) {
		float *b = data;
		const Instrument* ins = g_instruments + i; 
		int noteLenSamples = ins->noteLen * SAMPLE_FREQUENCY_01;
		int oscType = ins->oscType;
		float attackTime = ins->attackTime * 0.01f;
		float decayTime = ins->decayTime * 0.01f;
		float sustainTime = ins->sustainTime * 0.01f;

		// Loop through all rows for given instrument
		for ( int r = 0; r < ROW_COUNT; r++, b += ROW_SIZE_SAMPLES ) {
			g_patternNo = ( r >> 4 );
			g_patternRowNo = ( r & 0xF );
			char note = GetFixedOrStreamValue ( ins->note );
			if ( !note )
				continue;
			note += ins->noteOffset;
			float f = NoteToFrequency ( note );
			float f1 = f;
			if ( ins->noteEnd )
				f1 = NoteToFrequency ( ins->noteEnd + ins->noteOffset );
			float filtF = GetFixedOrStreamValue ( ins->filterCutoff ) * 0.01f;
			float filtQ = ins->filterRes * 0.01f;
			float filtMod = ins->filterCutoffModulation * 0.01f;
			buf0LPRes = buf1LPRes = 0;
			float vol = ins->volume * 0.01f;
			float wrapLevel = ins->wrapLevel * 0.01f;
			float hardLevel = ins->hardLevel * 0.01f;
			float compLow = ins->compressorLow * 0.01f;
			float compHigh = ins->compressorHigh * 0.01f;
			float compDL = compHigh - compLow;
			float *b1 = b;

			// Create one sound sample based on instrument and sequence parameters
			for ( int k = 0; k < noteLenSamples; k++, b1++ ) {
				float t1 = (float) k / noteLenSamples;
				float t = (float) k / SAMPLE_FREQUENCY * ( f + (f1-f)*t1 );
				float y;
				if ( oscType == 0 )
					y = sinf ( t );
				if ( oscType == 1 )
					y = Sawtooth ( t );
				if ( oscType == 2 )
					y = Square ( t );
				if ( oscType == 3 )
					y = (float)rand ( ) / RAND_MAX - 0.5f;
				if ( t1 < attackTime )
					y *= t1 / attackTime;
				else if ( t1 < decayTime )
					y *= 1.0f - 0.3f * ( t1 - attackTime ) / ( decayTime - attackTime );
				else if ( t1 < sustainTime )
					y *= 0.7f;
				else
					y *= 0.7f * ( 1.0f - t1  ) / ( 1.0f - sustainTime );
				if ( filtF )
					y = LowPassResonantFilter ( y, filtF + filtMod * t1, filtQ );
				if ( wrapLevel )
					while ( abs ( y ) > wrapLevel )
						y = 2.0f * wrapLevel * Sign ( y ) - y;
				if ( compHigh ) {
					float v = fabs ( y ) - compLow;
					if ( v > 0.0f ) {
						v = compHigh - compDL / ( 1.0f + 10.0f * v / compDL );
						y = v * Sign ( y );
					}
				}
				*b1 += y * vol;
				if ( ins->echoRepeats ) {
					float c = ins->echoDamping*0.01f;
					for ( int kk = 0; kk < ins->echoRepeats; kk++ ) {
						*(b1+(DWORD)((kk+1)*ins->echoDelay*(0.01f*SAMPLE_FREQUENCY))) += y * vol * c;
						c *= c;
					}
				}
			}
		}
	}

	// Copy just generated sound to audio buffer and play it
	short *s = finalBuf;
	for ( int i = 0; i < PLAY_TIME*SAMPLE_FREQUENCY; i++ )
		*(s++) = (short) ( 32000.0f * (*(data++)) * MASTER_VOLUME );
	WAVEHDR hdr = {0};
	hdr.lpData = (LPSTR) finalBuf;
	hdr.dwBufferLength  = PLAY_TIME*SAMPLE_FREQUENCY*sizeof(short);
	waveOutPrepareHeader ( g_wo, &hdr, sizeof(hdr) );
	waveOutWrite ( g_wo, &hdr, sizeof(hdr) );
}